package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Toolkit;
import java.awt.event.MouseEvent;

import javax.swing.AbstractButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JScrollPane;


/**
 * Abstract data container class that holds most of the data necessary for jfcUnit to fire mouse events.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public abstract class AbstractMouseEventData extends AbstractEventData {
    /**
     * boolean specifying whether this event will show a popup.
     */
    private boolean m_isPopupTrigger;

    /**
     * Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks).
     */
    private int m_numberOfClicks = 0;

    /**
     * Get the default modifiers for the popup trigger.
     *
     * @param popupTrigger true if to enable the popup menu.
     * @return int popup modifiers else DEFAULT_MOUSE_MODIFIERS.
     */
    public static int getDefaultModifiers(final boolean popupTrigger) {
        if (popupTrigger) {
            return MouseEvent.BUTTON3_MASK;
        }

        return DEFAULT_MOUSE_MODIFIERS;
    }

    /**
     * Get the default popup trigger.
     * @return int popup modifers else DEFAULT_MOUSE_MODIFIERS
     */
    public int getDefaultModifiers() {
        return getDefaultModifiers(getPopupTrigger());
    }

    /**
     * Get the string representing the current modifiers.
     * @return String version of the modifiers.
     */
    public final String getModifierText() {
        StringBuffer buf       = new StringBuffer();
        int          modifiers = getModifiers();

        if ((modifiers & MouseEvent.ALT_MASK) != 0) {
            buf.append(Toolkit.getProperty("AWT.alt", "Alt"));
            buf.append("+");
        }

        // Meta should only be available to Key events.
        //       if ((modifiers & MouseEvent.META_MASK) != 0) {
        //           buf.append(Toolkit.getProperty("AWT.meta", "Meta"));
        //           buf.append("+");
        //       }
        if ((modifiers & MouseEvent.CTRL_MASK) != 0) {
            buf.append(Toolkit.getProperty("AWT.control", "Ctrl"));
            buf.append("+");
        }

        if ((modifiers & MouseEvent.SHIFT_MASK) != 0) {
            buf.append(Toolkit.getProperty("AWT.shift", "Shift"));
            buf.append("+");
        }

        if ((modifiers & MouseEvent.ALT_GRAPH_MASK) != 0) {
            buf.append(Toolkit.getProperty("AWT.altGraph", "Alt Graph"));
            buf.append("+");
        }

        if ((modifiers & MouseEvent.BUTTON1_MASK) != 0) {
            buf.append(Toolkit.getProperty("AWT.button1", "Button1"));
            buf.append("+");
        }

        if ((modifiers & MouseEvent.BUTTON2_MASK) != 0) {
            buf.append(Toolkit.getProperty("AWT.button2", "Button2"));
            buf.append("+");
        }

        if ((modifiers & MouseEvent.BUTTON3_MASK) != 0) {
            buf.append(Toolkit.getProperty("AWT.button3", "Button3"));
            buf.append("+");
        }

        if (buf.length() > 0) {
            buf.setLength(buf.length() - 1); // remove trailing '+'
        }

        return buf.toString();
    }

    /**
     * Set the attribute value.
     *
     * @param numberOfClicks The new value of the attribute
     */
    public final void setNumberOfClicks(final int numberOfClicks) {
        m_numberOfClicks = numberOfClicks;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute
     */
    public final int getNumberOfClicks() {
        return m_numberOfClicks;
    }

    /**
     * Return the default number of clicks.
     * @return defaut number of clicks.
     */
    public int getDefaultNumberOfClicks() {
        return EventDataConstants.DEFAULT_NUMBEROFCLICKS;
    }

    /**
     * Get the default Popup trigger setting.
     * @return default popup trigger setting.
     */
    public boolean getDefaultPopupTrigger() {
        return EventDataConstants.DEFAULT_ISPOPUPTRIGGER;
    }

    /**
     * Set the attribute value.
     *
     * @param isPopupTrigger The new value of the attribute
     */
    public void setPopupTrigger(final boolean isPopupTrigger) {
        m_isPopupTrigger = isPopupTrigger;
    }

    /**
     * Get the attribute value.
     *
     * @return boolean    The value of the attribute
     */
    public boolean getPopupTrigger() {
        return m_isPopupTrigger;
    }

    /**
     * Check if this event can consume the {@link java.awt.AWTEvent}.
     *
     * @param ae {@link java.awt.AWTEvent} to be consumed.
     * @return boolean true if the event can be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if (!(ae instanceof MouseEvent)) {
            return false;
        }

        MouseEvent me = (MouseEvent) ae;

        if (me.getID() == MouseEvent.MOUSE_DRAGGED) {
            return false;
        }

        Object source = ae.getSource();

        if (source instanceof JScrollPane) {
            return false;
        }

        if (source instanceof JComponent) {
            String name = ((JComponent) source).getName();

            if ((name != null) && name.endsWith(".glassPane")) {
                return false;
            }
        }

        if (source instanceof AbstractButton) {
            if (((AbstractButton) source).getParent() instanceof JComboBox) {
                return false;
            }
        }

        if (ae.getSource().equals(getRoot((Component) ae.getSource()))) {
            return false;
        }

        return true;
    }

    /**
     * Get the attribute value.
     *
     * @param ae {@link java.awt.Event} to be processed.
     * @return boolean    The value of the attribute
     */
    public boolean consume(final AWTEvent ae) {
        MouseEvent me = (MouseEvent) ae;
        int        id = me.getID();

        if ((id == MouseEvent.MOUSE_MOVED)
                || (id == MouseEvent.MOUSE_ENTERED)
                || (id == MouseEvent.MOUSE_EXITED)
                || (isValid() && (ae.getSource() == getRoot()))) {
            // Ignore the event.
            return true;
        }

        return false;
    }

    /**
     * Compare to event datas and determine if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        AbstractMouseEventData data = (AbstractMouseEventData) o;

        return (data.getNumberOfClicks() == getNumberOfClicks())
            && (data.getPopupTrigger() == getPopupTrigger());
    }

    /**
     * hashCode is delegated to the super class.
     * @return hashCode of the super class.
     */
    public int hashCode() {
        return super.hashCode();
    }

    /**
     * Populate the element given with the attributes of the
     * event.
     *
     * @param e element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);

        if (getNumberOfClicks() != getDefaultNumberOfClicks()
            && getNumberOfClicks() > 0) {
            e.setAttribute(JFCXMLConstants.CLICKS, "" + getNumberOfClicks());
        }

        boolean trigger = getPopupTrigger();

        if (trigger != getDefaultPopupTrigger()) {
            String value = "true";

            if (!trigger) {
                value = "false";
            }

            e.setAttribute(JFCXMLConstants.POPUPTRIGGER, value);
        }
    }

    /**
     * Check if the event has the same source as this event data.
     *
     * @param ae {@link java.awt.AWTEvent} to be checked.
     * @return true if the events have the same source.
     */
    public boolean sameSource(final AWTEvent ae) {
        // REDTAG: Is this check incorrect?
        if (isValid()) {
            return ((Component) ae.getSource() == getComponent());
        } else {
            return true;
        }
    }

    /**
     * Return a string representing the eventdata.
     *
     * @return String description of the event data.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" clicks: " + getNumberOfClicks());
        buf.append(" popup: " + getPopupTrigger());

        return buf.toString();
    }
}
