package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.TestHelper;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 * This class is specific to events on a {@link javax.swing.JComboBox}.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JMenuMouseEventData extends AbstractMouseEventData {
    /**
     * The {@link JMenuBar} on which to trigger the event.
     */
    private JComponent m_menu;

    /**
     * The zero-based index of the specific menu elements on which to trigger the event.
     */
    private int[] m_elementIndexes;

    /**
     * Constructor.
     */
    public JMenuMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param menu  The {@link JMenu} on which to trigger the event.
     * @param elementIndexes
     *                   The zero-based index of the menu items to select.
     */
    public JMenuMouseEventData(final JFCTestCase testCase,
        final JComponent menu, final int[] elementIndexes) {
        this(testCase, menu, elementIndexes, DEFAULT_NUMBEROFCLICKS,
            DEFAULT_MOUSE_MODIFIERS, DEFAULT_ISPOPUPTRIGGER, DEFAULT_SLEEPTIME);
        setValid(true);
    }

    /**
     * Constructor.
     *
     * @param testCase  The {@link JFCTestCase} on whose thread <code>awtSleep()</code> has to be invoked.
     * @param menu  The {@link JMenu} on which to trigger the event.
     * @param elementIndexes
     *                   The zero-based index of the menu items to select.
     * @param numberOfClicks
     *                   Number of clicks in the {@link MouseEvent} (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JMenuMouseEventData(final JFCTestCase testCase,
        final JComponent menu, final int[] elementIndexes,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        setTestCase(testCase);
        setSource(menu);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setElementIndexes(elementIndexes);
        setSleepTime(sleepTime);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param menu The new value of the attribute.
     */
    public void setSource(final JComponent menu) {
        m_menu = menu;
    }

    /**
     * Get the attribute value.
     *
     * @return JComboBox    The value of the attribute.
     */
    public final JComponent getSource() {
        return m_menu;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Set the attribute value.
     *
     * @param elementIndexes The new value of the attribute.
     */
    public final void setElementIndexes(final int[] elementIndexes) {
        m_elementIndexes = elementIndexes;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute.
     */
    public final int[] getElementIndexes() {
        return m_elementIndexes;
    }

    /**
     * Returns true if the event can be consumed by
     * this instance of event data.
     *
     * @param ae Event to be consumed.
     * @return true if the event was consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        Object source = ae.getSource();

        if (!super.canConsume(ae)
                || !((source instanceof JMenuItem)
                || (source instanceof JPopupMenu)
                || (source instanceof JMenuBar))) {
            return false;
        }

        return true;
    }

    /**
     * Consume the event.
     *
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        Object source = ae.getSource();

        if (source instanceof JMenu) {
            return true;
        }

        if (super.consume(ae)) {
            return true;
        }

        JMenuItem item = (JMenuItem) source;
        PathData  data = new PathData(item);
        Object    root = data.getRoot(item);

        if (root instanceof JMenuBar) {
            JMenuBar bar = (JMenuBar) root;
            setSource(bar);
            setElementIndexes(data.getIndexes(bar));
        } else if (root instanceof JPopupMenu) {
            JPopupMenu popup = (JPopupMenu) root;
            setSource(popup);
            setElementIndexes(data.getIndexes(popup));
        }

        //        JMenuBar bar = (JMenuBar)data.getRoot(item);
        //        setSource(bar);
        setPosition(CENTER);
        setReferencePoint(null);
        setSleepTime(getDefaultSleepTime());
        setValid(true);

        return true;
    }

    /**
     * Compare to event datas and determine if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JMenuMouseEventData data = (JMenuMouseEventData) o;

        return (data.getSource() == getSource())
            && (data.getElementIndexes() == getElementIndexes());
    }

    /**
     * Get the hashCode.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode();
    }

    /**
     * Populate the JComboBoxMouseEventData XML element.
     * @param e element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);

        StringBuffer buf = new StringBuffer();

        for (int i = 0; i < this.m_elementIndexes.length; i++) {
            buf.append(m_elementIndexes[i]);
            buf.append(",");
        }

        buf.setLength(buf.length() - 1);
        e.setAttribute(
            JFCXMLConstants.INDEXES,
            buf.toString());
        e.setAttribute(JFCXMLConstants.TYPE, "JMenuMouseEventData");
    }

    /**
     * Prepare the {@link Component} to receive the event.
     *
     * @return true if the component is ready to receive the event.
     */
    public boolean prepareComponent() {
        JFCTestCase tc = getTestCase();
        TestHelper  th = tc.getHelper();
        tc.resumeAWT();

        JComponent m    = m_menu;
        Component  comp = m.getComponent(m_elementIndexes[0]);
        th.enterClickAndLeave(new MouseEventData(tc, comp, 1));

        for (int i = 1; i < m_elementIndexes.length; i++) {
            if (comp instanceof JMenu) {
                JMenu menu = (JMenu) comp;
                comp = menu.getMenuComponent(m_elementIndexes[i]);
                th.enterClickAndLeave(
                    new MouseEventData(
                        getTestCase(),
                        comp,
                        0,
                        0));
                tc.flushAWT();

                if (i != (m_elementIndexes.length - 1)) {
                    JPopupMenu pm = menu.getPopupMenu();
                    int popupLeft   = pm.getLocationOnScreen().x;
                    int popupRight  = popupLeft + pm.getSize().width;
                    int popupTop    = pm.getLocationOnScreen().y;
                    int popupBottom = popupTop + pm.getSize().height;
                    int left        = menu.getLocationOnScreen().x;
                    int right       = left + menu.getSize().width;
                    int top         = menu.getLocationOnScreen().y;
                    int bottom      = top + menu.getSize().height;

                    //              Rectangle popupBounds = pm.getBounds();
                    //              Rectangle bounds = menu.getBounds();
                    //              // check for horizontal move to submenu
                    //              // Off to the right
                    //              int right = bounds.x + bounds.width;
                    //              int top = bounds.y;
                    //              int bottom = bounds.y + bounds.height;
                    //              int left = bounds.x;
                    //              int popupRight = popupBounds.x + popupBounds.width;
                    //              int popupLeft = popupBounds.x;
                    //              int popupTop = popupBounds.y;
                    //              int popupBottom = popupBounds.y + popupBounds.height;
                    if (right < popupRight) {
                        // Move to the right edge
                        th.enterClickAndLeave(
                            new MouseEventData(
                                getTestCase(),
                                comp,
                                0,
                                0,
                                false,
                                0,
                                EAST));
                    } else if (popupLeft < left) {
                        // Off to the left
                        th.enterClickAndLeave(
                            new MouseEventData(
                                getTestCase(),
                                comp,
                                0,
                                0,
                                false,
                                0,
                                WEST));
                    } else if (top < popupTop) {
                        // Below
                        // Move to the right edge
                        th.enterClickAndLeave(
                            new MouseEventData(
                                getTestCase(),
                                comp,
                                0,
                                0,
                                false,
                                0,
                                SOUTH));
                    } else if (popupBottom < bottom) {
                        // Above
                        th.enterClickAndLeave(
                            new MouseEventData(
                                getTestCase(),
                                comp,
                                0,
                                0,
                                false,
                                0,
                                NORTH));
                    }

                    //              tc.flushAWT();
                }
            }
        }

        th.enterClickAndLeave(new MouseEventData(
                getTestCase(),
                comp));
        getTestCase().flushAWT();

        return false;
    }

    /**
     * Get string description of event.
     *
     * @return String description of event.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" index: " + getElementIndexes());

        return buf.toString();
    }
}
