package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import javax.swing.table.JTableHeader;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 * This class is specific to events on a JTableHeader.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JTableHeaderMouseEventData extends AbstractMouseEventData {
    /**
     * The JTableHeader on which to trigger the event.
     */
    private JTableHeader m_header;

    /**
     * The zero-based column index of the specific cell on which to trigger the event.
     */
    private int m_columnIndex;

    /**
     * Constructor.
     */
    public JTableHeaderMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param header   The component on which to trigger the event.
     * @param columnIndex
     *                  The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JTableHeaderMouseEventData(final JFCTestCase testCase,
        final JTableHeader header, final int columnIndex,
        final int numberOfClicks) {
        this(testCase, header, columnIndex, numberOfClicks, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param header    The component on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTableHeaderMouseEventData(final JFCTestCase testCase,
        final JTableHeader header, final int columnIndex,
        final int numberOfClicks, final long sleepTime) {
        this(testCase, header, columnIndex, numberOfClicks,
            DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param header   The component on which to trigger the event.
     * @param columnIndex
     *                  The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public JTableHeaderMouseEventData(final JFCTestCase testCase,
        final JTableHeader header, final int columnIndex,
        final int numberOfClicks, final boolean isPopupTrigger) {
        this(testCase, header, columnIndex, numberOfClicks, isPopupTrigger,
            DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param header    The component on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column of index the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTableHeaderMouseEventData(final JFCTestCase testCase,
        final JTableHeader header, final int columnIndex,
        final int numberOfClicks, final boolean isPopupTrigger,
        final long sleepTime) {
        this(testCase, header, columnIndex, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param header    The component on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTableHeaderMouseEventData(final JFCTestCase testCase,
        final JTableHeader header, final int columnIndex,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, header, columnIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code>
     *                  has to be invoked.
     * @param header    The component on which to trigger the event.
     * @param columnIndex
     *                  The zero-based column index of the specific cell
     *                  on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click,
     *                  2 for double clicks)
     * @param modifiers The modifier key values that need to be passed
     *                  onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JTableHeaderMouseEventData(final JFCTestCase testCase,
        final JTableHeader header, final int columnIndex,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, header, columnIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param header    The component on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                   The CUSTOM mouse position within the cell.
     */
    public JTableHeaderMouseEventData(final JFCTestCase testCase,
        final JTableHeader header, final int columnIndex,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, header, columnIndex, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param header    The component on which to trigger the event.
     * @param columnIndex
     *                   The zero-based column index of the specific cell on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If _position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the _position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
     *                                   Otherwise, the _referencePoint is unused.
     */
    public JTableHeaderMouseEventData(final JFCTestCase testCase,
        final JTableHeader header, final int columnIndex,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        setTestCase(testCase);
        setSource(header);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setColumnIndex(columnIndex);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param columnIndex The new value of the attribute.
     */
    public final void setColumnIndex(final int columnIndex) {
        m_columnIndex = columnIndex;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute.
     */
    public final int getColumnIndex() {
        return m_columnIndex;
    }

    /**
     * Set the attribute value.
     *
     * @param header The new value of the attribute.
     */
    public final void setSource(final JTableHeader header) {
        m_header = header;
    }

    /**
     * Get the attribute value.
     *
     * @return JTableHeader    The value of the attribute.
     */
    public final JTableHeader getSource() {
        return m_header;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Check if this event can consume the event given.
     * @param ae AWTEvent to be consumed.
     * @return true if the event may be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if ((ae.getSource() instanceof JTableHeader)
                && super.canConsume(ae)
                && sameSource(ae)) {
            return true;
        }

        return false;
    }

    /**
     * Consume the event.
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        MouseEvent   me     = (MouseEvent) ae;
        JTableHeader source = (JTableHeader) me.getSource();
        setSource(source);
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());

        Point p = new Point(
                me.getX(),
                me.getY());
        Point screen = source.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);
        setSleepTime(getDefaultSleepTime());

        int column = source.columnAtPoint(p);

        setColumnIndex(column);
        setPosition(CENTER);
        setReferencePoint(null);

        setValid(true);

        return true;
    }

    /**
     * Compare to event datas and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JTableHeaderMouseEventData data = (JTableHeaderMouseEventData) o;

        return (data.getColumnIndex() == getColumnIndex());
    }

    /**
     * Get the hashCode.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode() + m_columnIndex;
    }

    /**
     * Populate the XML Element with this objects attributes.
     * @param e element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "JTableHeaderMouseEventData");
        e.setAttribute(JFCXMLConstants.COLUMN, "" + getColumnIndex());
    }

    /**
     * Prepare the component to receive the event.
     *
     * @return true if the component is ready to receive the event.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource())) {
            return false;
        }

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
        }

        Rectangle headerRect = m_header.getHeaderRect(m_columnIndex);

        if (testCase != null) {
            // try to clear the event queue
            testCase.pauseAWT();
        }

        Point p = calculatePoint(headerRect);

        if (!m_header.getVisibleRect().contains(p)) {
            // Center the new point in the visible area.
            Rectangle tvr  = m_header.getTable().getVisibleRect();
            Rectangle rect = new Rectangle(p.x - (int) (tvr.getWidth() / 2),
                    (int) tvr.getY(), (int) tvr.getWidth(),
                    (int) tvr.getHeight());
            m_header.getTable().scrollRectToVisible(rect);
        }

        Point screen = m_header.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        return true;
    }

    /**
     * Get string description of event.
     * @return String description of event.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" column: " + getColumnIndex());

        return buf.toString();
    }
}
