package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.TestHelper;
import junit.extensions.jfcunit.keyboard.JFCKeyStroke;
import junit.extensions.jfcunit.keyboard.KeyMapping;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.event.KeyEvent;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class StringEventData extends AbstractKeyEventData {
    /**
     * The string code to be fired.
     */
    private Component m_comp = null;

    /**
     * The string code to be fired.
     */
    private String m_string = "";

    /**
     * Default Constructor.
     */
    public StringEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code>
     * has to be invoked.
     * @param comp     The component on which to trigger the event.
     * @param string   The string to be sent to the component
     */
    public StringEventData(final JFCTestCase testCase, final Component comp,
        final String string) {
        this(testCase, comp, string, DEFAULT_KEY_MODIFIERS, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param string    The string to be sent to the component
     * @param sleepTime The wait time in ms between each event.
     */
    public StringEventData(final JFCTestCase testCase, final Component comp,
        final String string, final long sleepTime) {
        this(testCase, comp, string, DEFAULT_KEY_MODIFIERS, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param comp      The component on which to trigger the event.
     * @param string    The string to be sent to the component
     * @param modifiers The modifier string values that need to be passed onto the event.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public StringEventData(final JFCTestCase testCase, final Component comp,
        final String string, final int modifiers, final long sleepTime) {
        setTestCase(testCase);
        setSource(comp);
        setString(string);
        setModifiers(modifiers);
        setSleepTime(sleepTime);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param comp   The new value of the attribute.
     */
    public final void setSource(final Component comp) {
        m_comp = comp;
    }

    /**
     * Get the attribute value.
     *
     * @return Component    The value of the attribute.
     */
    public final Component getSource() {
        return m_comp;
    }

    /**
     * Set the string to be sent.
     *
     * @param string Key code.
     */
    public final void setString(final String string) {
        m_string = string;
        setupKeyStrokes();
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Get the string to be sent.
     *
     * @return String string to be sent.
     */
    public String getString() {
        return m_string;
    }

    /**
     * Check if this event can consume the AWTEvent.
     *
     * @param ae Event to be consumed.
     * @return boolean true if the event can be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if (!(ae instanceof KeyEvent)) {
            return false;
        }

        // If not a defined keystoke then cannot consume by the string.
        JFCKeyStroke[] stroke = getKeyMapping().getKeyStrokes(
                ((KeyEvent) ae).getKeyChar());
        char           c = ((KeyEvent) ae).getKeyChar();

        if (Character.isISOControl(c)) {
            return false;
        }

        if (isValid()) {
            int mods = ((KeyEvent) ae).getModifiers();

            if (getModifiers() != mods) {
                return false;
            }
        }

        return ((stroke != null) && (stroke.length > 0))
        && super.canConsume(ae)
        && (isMetaChar(((KeyEvent) ae).getKeyCode())
        || (((KeyEvent) ae).getKeyChar() != KeyEvent.CHAR_UNDEFINED));
    }

    /**
     * Consume the event.
     * @param ae Event to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        setSource((Component) ae.getSource());

        StringBuffer buf = new StringBuffer();

        if (isValid()) {
            buf.append(getString());
        }

        buf.append(((KeyEvent) ae).getKeyChar());
        setSleepTime(getDefaultSleepTime());

        int mods = ((KeyEvent) ae).getModifiers();

        super.setModifiers(mods);
        setString(buf.toString());
        setValid(true);

        return true;
    }

    /**
     * Populate the element with the key code.
     * @param e Element to be populated with the data for this event.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute("string", m_string);
    }

    /**
     * Generate a description of the event.
     *
     * @return String description of the event.
     */
    public String toString() {
        return "StringEventData:" + getString() + " on " + getComponent() + " "
        + getModifiers() + " " + getSleepTime();
    }

    /**
     * Setup the keystrokes for the event.
     */
    protected void setupKeyStrokes() {
        KeyMapping km = TestHelper.getKeyMapping();
        clearKeyStrokes();

        for (int i = 0; i < m_string.length(); i++) {
            JFCKeyStroke[] strokes = km.getKeyStrokes(m_string.charAt(i));
            addKeyStrokes(strokes);
        }

        applyModifier(getModifiers());
    }
}
