package junit.extensions.jfcunit.xml;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.JFCTestHelper;
import junit.extensions.jfcunit.RobotTestHelper;

import junit.extensions.xml.IXMLProcedure;
import junit.extensions.xml.IXMLTest;
import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.XMLObjectCache;
import junit.extensions.xml.XMLTagResourceBundle;
import junit.extensions.xml.XMLTestSuite;
import junit.extensions.xml.XMLUtil;

import junit.framework.Test;

import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.NodeList;

import java.awt.AWTException;


/**
 * This class will handle the processing of &lt;test&gt; nodes.
 *
 * <h3>Description</h3>
 * <p>
 *   This tag handler invokes the test case given.
 * </p>
 *
 * <h3>Attributes</h3>
 * <table border="1" cellpadding="2" cellspacing="0">
 *   <tr>
 *     <td valign="top"><b>Attribute</b></td>
 *     <td valign="top"><b>Description</b></td>
 *     <td align="center" valign="top"><b>Required</b></td>
 *     <td valign="top"><b>Default</b></td>
 *     <td valign="top"><b>Values</b></td>
 *   </tr>
 *   <tr>
 *     <td valign="top">name</td>
 *     <td valign="top">name of the test case</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">Alpha Numeric Name</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">robot</td>
 *     <td valign="top">Use a robot or send events directly to AWT Event queue.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">false</td>
 *     <td valign="top">true if events are to be sent via the Robot</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">assertexit</td>
 *     <td valign="top">
 *         Assert the System.exit() command, ExitException will be
 *         thrown by the security manager and the application will not exit.
 *     </td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">false</td>
 *     <td valign="top">true if the system exist should be asserted.</td>
 *   </tr>
 * </table>
 * <h3>Example</h3>
 * <pre>
 * &lt;test name="Login"&gt
 * ...
 * &lt;/test&gt;
 * &lt;test name="Logout" assertexit="true"&gt;
 * ...
 * &lt;/test&gt;
 * </pre>
 * <p>
 * The above runs the defined testcase.
 * </p>
 * @author Kevin Wilson
 */
public class JFCXMLTestCase extends JFCTestCase implements IXMLTestCase,
    JFCXMLConstants {
    /**
     * The element to be processed.
     */
    private Element m_element;

    /**
     * The name of the test xml script file - found from the classpath.
     */
    private String m_filename;

    /**
     * Parent test case.
     */
    private Test m_parent;

    /**
     * Map of procedures.
     */
    private XMLObjectCache m_procedures = new XMLObjectCache();

    /**
     * A Map of all objects that have been found - keyed by the string name.
     */
    private XMLObjectCache m_properties = new XMLObjectCache();

    /**
     * The default constructor that is needed to createa a test case.
     *
     * @param filename    The name of the test xml script file (found from the classpath)
     * @param element     The Element to be processed
     */
    public JFCXMLTestCase(final String filename, final Element element) {
        super(filename);
        m_filename     = filename;
        m_element      = element;

        boolean robot  = XMLUtil.getBooleanAttributeValue(element, ROBOT);

        if (robot) {
            try {
                setHelper(new RobotTestHelper());
            } catch (AWTException ex) {
                throw new RuntimeException("Could not create robot." + ex);
            }
        } else {
            setHelper(new JFCTestHelper());
        }

        boolean assertExit = XMLUtil.getBooleanAttributeValue(element,
                ASSERTEXIT);
        setAssertExit(assertExit);
    }

    /**
     * Get the debug state.
     * @return boolean true if debugging has been enabled.
     */
    public final boolean getDebug() {
        return (m_properties.get(DEBUG) != null);
    }

    /**
     * Set the parent test case.
     * @param parent Parent test case.
     */
    public final void setParent(final IXMLTest parent) {
        this.m_parent = parent;
        m_properties.setParent(((IXMLTest) parent).getPropertyCache());
        m_procedures.setParent(((IXMLTest) parent).getProcedureCache());
    }

    /**
     * Get the procedure withthe given name.
     * @param name Name of the procedure to return.
     * @return IXMLProcedure referenced by name.
     */
    public final IXMLProcedure getProcedure(final String name) {
        return (IXMLProcedure) m_procedures.get(name);
    }

    /**
     * Returns the cahce of procedures.
     * @return XMLObjectCache cache of procedures.
     */
    public final XMLObjectCache getProcedureCache() {
        return m_procedures;
    }

    /**
     * Retrieve the object that was found previously.
     *
     * @param name    The name of the object that was found
     * @return    The object to be retrieved
     */
    public final Object getProperty(final String name) {
        return m_properties.get(name);
    }

    /**
     * Get the property cache.
     * @return XMLObjectCache used to hold the properties.
     */
    public final XMLObjectCache getPropertyCache() {
        return m_properties;
    }

    /**
     * Get the name of a component which has been found.
     * @param comp Component to locate
     * @return The name of the component.
     */
    public final String getPropertyName(final Object comp) {
        return m_properties.getName(comp);
    }

    /**
     * Get the names of the properties.
     * @return array containing the names of the properties.
     */
    public final String[] getPropertyNames() {
        return m_properties.getNames();
    }

    /**
     * Add a procedure to the test case.
     * @param proc Procedure to be added.
     */
    public final void addProcedure(final IXMLProcedure proc) {
        m_procedures.put(
            proc.getName(),
            proc);
    }

    /**
     * Call the procedure with the given name.
     * @param name Name of the procedure.
     * @param callElement Element which contains the attributes to
     * be passed to the procedures.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public final void callProcedure(final String name, final Element callElement)
        throws XMLException {
        XMLObjectCache cache = m_properties;
        m_properties = new XMLObjectCache();
        m_properties.setParent(cache);

        if (callElement != null) {
            NamedNodeMap atts = callElement.getAttributes();
            int          size = atts.getLength();

            for (int i = 0; i < size; i++) {
                Attr   node = (Attr) atts.item(i);
                String an = node.getName();

                if (!CALL.equals(an)) {
                    String value = XMLUtil.getAttribute(callElement, an);
                    value = this.resolveProperties(value);
                    addProperty(an, value);
                }
            }
        }

        IXMLProcedure proc = getProcedure(name);

        if (proc == null) {
            m_properties.setParent(null);
            m_properties = cache;

            if ("setUp".equals(name) || "tearDown".equals(name)) {
                return;
            }

            throw new XMLException("Procedure not found:" + name, null,
                m_element,
                getPropertyCache());
        }

        this.processChildren(proc.getElement());
        m_properties.setParent(null);
        m_properties = cache;
    }

    /**
     * Process the child XML elements.
     * @param element Element which contains the children to
     * be processed.
     * @throws XMLException may be thrown.
     */
    public final void processChildren(final Element element)
        throws XMLException {
        // Get the children and add to the suite.
        NodeList children = element.getChildNodes();
        Element  child;

        for (int i = 0; i < children.getLength(); i++) {
            if (children.item(i) instanceof Element) {
                child = (Element) children.item(i);

                String name = child.getTagName();

                try {
                    XMLTagResourceBundle.getTagHandler(child, this, name)
                                        .processElement();
                } catch (XMLException xe) {
                    throw xe;
                } catch (Throwable t) {
                    // Wrap into a XMLException and throw.
                    XMLException x = new XMLException("Exception ("
                            + t.toString() + " from tag: " + name, t, child,
                            getPropertyCache());
                    throw x;
                }

                if (child != children.item(i)) {
                    for (;
                            (child != children.item(i))
                            && (i < children.getLength()); i++) {
                        ;
                    }

                    if (i == children.getLength()) {
                        throw new junit.extensions.xml.XMLException("Lost where we were at current node was removed.",
                            null,
                            child,
                            getPropertyCache());
                    }
                }
            }
        }
    }

    /**
     * This method is the one that actually performs the test by processing the elements.
     *
     * @exception  Exception   An instance of java.lang.Exception can be thrown.
     */
    public final void runXMLTest() throws Exception {
        boolean debug = XMLUtil.getBooleanAttributeValue(m_element, DEBUG);

        if (debug) {
            m_properties.put(DEBUG, Boolean.TRUE);
        } else {
            m_properties.remove(DEBUG);
        }

        processChildren(m_element);
    }

    /**
     * Returns an empty suite.
     *
     * @return Test   An empty XMLTestSuite is returned by default
     * @exception  Exception   An instance of java.lang.Exception can be thrown.
     */
    public static Test suite() throws Exception {
        return new XMLTestSuite();
    }

    /**
     * Add each found object into the cache map.
     *
     * @param name    The name of the found object
     * @param obj     The actual object that was found
     */
    public final void addProperty(final String name, final Object obj) {
        m_properties.put(name, obj);
    }

    /**
     * For each test (method) element, the found objects has to be cleared.
     */
    public final void clearProperties() {
        m_properties.clear();
    }

    /**
     * Remove the property with the name given.
     *
     * @param name Name of the object to be removed
     */
    public final void removeProperty(final String name) {
        m_properties.remove(name);
    }

    /**
     * Resolve embeded property names withing a string.
     * @param s String to be resolved.
     * @return String without property names.
     */
    public String resolveProperties(final String s) {
        if (s == null) {
            return null;
        }

        int    index = -1;
        String str = new String(s);

        do {
            index = str.lastIndexOf("${");

            if (index != -1) {
                int    lindex = str.indexOf("}", index);

                String variableName = str.substring(index + 2, lindex);
                assertTrue("Evaluation of \"" + str + "\" Null Variable name.",
                    variableName.length() > 0);

                Object value = getProperty(variableName);

                if (value == null) {
                    System.err.println("WARNING: Evaluation of \"" + s
                        + "\" Variable not found assuming empty string for:"
                        + variableName);
                    value = "";
                }

                String valueString = value.toString();
                str = str.substring(0, index) + valueString
                    + str.substring(lindex + 1);
            }
        } while (index != -1);

        return str;
    }

    /**
     * Default setUp which does nothing. Override this to set up the environment
     * for your test
     *
     * @exception  Exception   An instance of java.lang.Exception can be thrown
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception {
        super.setUp();
        clearProperties();
        m_procedures.clear();
        loadProcedures(m_element);

        // Call the startup procedure
        callProcedure("setUp", null);
    }

    /**
     * Executes a test.
     *
     * @exception Throwable exceptions thrown by code.
     */
    protected void runTest() throws Throwable {
        runCode(
            new Runnable() {
                public void run() {
                    try {
                        runXMLTest();
                        flushAWT();
                    } catch (Exception exc) {
                        flushAWT();
                        setError(exc);
                    }
                }
            });
    }

    /**
     * Default tearDown which does nothing. Override this to tear down the
     * environment for your test.
     *
     * @exception  Exception   An instance of java.lang.Exception can be thrown
     * @see junit.framework.TestCase#setUp()
     */
    protected void tearDown() throws Exception {
        callProcedure("tearDown", null);
        super.tearDown();
    }

    /**
     * Process the child XML elements.
     * @param element Element which contains the children to
     * be processed.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    private void loadProcedures(final Element element) throws XMLException {
        // Get the children and add to the suite.
        NodeList children = element.getChildNodes();
        Element  child;

        for (int i = 0; i < children.getLength(); i++) {
            if (children.item(i) instanceof Element) {
                child = (Element) children.item(i);

                String name = child.getTagName();
                String call = child.getAttribute(CALL);

                if (PROCEDURE.equals(name) && (call.length() == 0)) {
                    XMLTagResourceBundle.getTagHandler(child, this, name)
                                        .processElement();
                }
            }
        }
    }
}
