package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;

import java.awt.Component;
import java.awt.Container;


/**
 * This class will handle the processing of &lt;dump&gt; tags.
 *
 * <H3>Required Attributes</H3>
 * refid
 *
 * <H3>Optional attributes</H3>
 *    recursive - Default vaue true
 *
 * @author <a href="mailto:baylward@nexagent.com">Bruce Aylward : Nexagent Ltd.</a>
 */
public class DumpTagHandler extends AbstractTagHandler {
    /**
     * Constructor for DumpTagHandler.
     *
     * @param element     The element to be processed.
     * @param testCase    The IXMLTestCase that uses this element.
     */
    public DumpTagHandler(final Element element, final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * Dump the object specified by the refid.
     * @throws XMLException may be thrown.
     */
    public void processElement() throws XMLException {
        validateElement();

        Object object = getXMLTestCase().getProperty(getRefId());

        if (object == null) {
            throw new XMLException("Referenced object could not be found",
                null,
                getElement(),
                getTest().getPropertyCache());
        } else if (!(object instanceof Component)) {
            throw new XMLException("Referenced object is not java.awt.Component",
                null,
                getElement(),
                getTest().getPropertyCache());
        }

        dumpComponents(
            (Component) object,
            1,
            getRecursive());
    }

    /**
     * Validate that the refid is specified.
     * @throws XMLException if the refid is not specified.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // check the element tag name
        checkElementTagName(DUMP);

        // reqd attribute: refid
        checkRequiredAttribute(REFID);
    }

    /**
     * Dumps the component and all its children to standard out.
     * @param parent The component to dump.
     * @param level The indentation level.
     * @param recursive True if the dump should be recursive.
     */
    private static void dumpComponents(final Component parent, final int level,
        final boolean recursive) {
        String indent = "";

        for (int t = 0; t < level; t++) {
            indent += "   ";
        }

        String name = parent.getName();

        if (name == null) {
            name = "<anonymous>";
        }

        System.out.println(indent + name + "(" + parent + ")");

        if (parent instanceof Container) {
            Component[] children = ((Container) parent).getComponents();

            for (int t = 0; t < children.length; t++) {
                if (recursive) {
                    dumpComponents(children[t], level + 1, recursive);
                } else {
                    name = children[t].getName();

                    if (name == null) {
                        name = "<anonymous>";
                    }

                    System.out.println(indent + name + "(" + children[t] + ")");
                }
            }
        }
    }

    /**
     * Returns the value of the RECURSIVE attribute for this element.
     * @return String  The value of the RECURSIVE attribute.
     */
    private boolean getRecursive() {
        return getBoolean(RECURSIVE, true);
    }

    /**
     * Returns the value of the REFID attribute for this element.
     * @return String  The value of the REFID attribute.
     */
    private String getRefId() {
        return getString(REFID);
    }
}
