package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;


/**
 * This is tag handler can measure the milliseconds within a
 * test case.
 *
 * <h3>Description</h3>
 * <p>
 * Mark the start of the process to be timed with a mark action.
 * <pre>
 * &lt;stopwatch id=&quot;myid&quot; action=&quot;mark&quot;/&gt;
 * </pre>
 *
 * Then afterward assert that the duration in milliseconds.
 * <pre>
 * &lt;stopwatch refid=&quot;myid&quot; action=&quot;lessthan&quot;
 *               value=&quot;8000&quot;/&gt;
 * </pre>
 * Or to simply write the duration to stdout.
 * <pre>
 * &lt;stopwatch refid=&quot;myid&quot; action=&quot;log&quot;/&gt;
 * </pre>
 * @see junit.extensions.jfcunit.eventdata.JComboBoxMouseEventData
 * @author Kevin Wilson
 */
public class StopWatchTagHandler extends AbstractTagHandler {
    /**
     * constructor.
     * @param element Element to be processed.
     * @param testCase containing test case.
     */
    public StopWatchTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * Get the action attribute from the element.
     * @return String The value of the action attribute.
     */
    public String getAction() {
        return getString(ACTION);
    }

    /**
     * Get the value of the id attribute.
     * @return String value of the id attribute.
     */
    public String getId() {
        return getString(ID);
    }

    /**
     * Get the value of the refid attribute.
     * @return String value of the refid attribute.
     */
    public String getRefid() {
        return getString(REFID);
    }

    /**
     * Process the element.
     * @throws XMLException may be thrown.
     */
    public void processElement() throws XMLException {
        long newMark = System.currentTimeMillis();
        validateElement();

        String action = getAction();

        if (MARK.equals(action)) {
            String id   = getId();
            Long   mark = new Long(newMark);
            getXMLTestCase().addProperty(id, mark);
        } else {
            //Get the stored objects
            String refid = getRefid();
            long   mark = ((Long) getXMLTestCase().getProperty(refid))
                .longValue();

            long duration = newMark - mark;

            if (action.equals(LOG)) {
                String name = this.getTestCase().getName();
                System.out.println("Duration of test(" + name + "/"
                    + getRefid() + ") = " + duration);
            }

            if (action.equals(LESSTHAN)) {
                long value = getLong(VALUE, 0);
                getTestCase().assertTrue("Duration exceeded.", duration < value);
            }
        }
    }

    /**
     * Validate that the element is properly configured.
     * @throws XMLException Exception may be thrown if there
     * are missing elements.
     */
    public void validateElement() throws XMLException {
        // check the element tag name
        checkElementTagName(STOPWATCH);

        // action is a required attribute
        checkRequiredAttribute(ACTION);

        String action = getAction();

        if (LESSTHAN.equals(action)) {
            // if action is not a mark, then a value is required.
            checkRequiredAttribute(
                getElement(),
                VALUE);
            checkRequiredAttribute(
                getElement(),
                REFID);
        } else if (LOG.equals(action)) {
            checkRequiredAttribute(
                getElement(),
                REFID);
        } else if (MARK.equals(action)) {
            checkRequiredAttribute(
                getElement(),
                ID);
        } else {
            throw new XMLException("Invalid action", null,
                getElement(),
                getTest().getPropertyCache());
        }
    }
}
