package junit.extensions.jfcunit.eventdata;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;

import java.util.StringTokenizer;

import javax.swing.JComponent;


/**
 * This is a extension of the click and drag elements. Where the
 * type specified is JMenuMouseEventData.
 *
 * <h3>Description</h3>
 * <p>
 *   Nothing is invoked directly by this tag handler. The click and drag elements
 *   use this tag handler to customize the data specificly for the JMenu.
 * </p>
 *
 * <h3>Parameters</h3>
 * <table border="1" cellpadding="2" cellspacing="0">
 *   <tr>
 *     <td valign="top"><b>Attribute</b></td>
 *     <td valign="top"><b>Description</b></td>
 *     <td align="center" valign="top"><b>Required</b></td>
 *     <td valign="top"><b>Default</b></td>
 *     <td valign="top"><b>Values</b></td>
 *   </tr>
 *   <tr>
 *     <td valign="top">type</td>
 *     <td valign="top">For this element the type must be set to JMenuMouseEventData</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">N/A</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">refid</td>
 *     <td valign="top">Id of a object reference which has been previously found.</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">N/A</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">node</td>
 *     <td valign="top">Yes</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">Must be string specifying the value of the node.</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">clicks</td>
 *     <td valign="top">The number of clicks to be executed.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">1</td>
 *     <td valign="top">Positive Integer</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">modifiers</td>
 *     <td valign="top">The modifiers to be used for this event.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">Button1 if popup is false otherwise Button3</td>
 *     <td valign="top">alt altgr control meta shift button1 button2 button3</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">popup</td>
 *     <td valign="top">True if this is event is using the popup button.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">false</td>
 *     <td valign="top">true false</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">sleeptime</td>
 *     <td valign="top">Maximum sleep time</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">500</td>
 *     <td valign="top">Positive Integer representing the milli-seconds</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">position</td>
 *     <td valign="top">Positioning of the mouse within the bounds of the component.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">center</td>
 *     <td valign="top">center north northeast east southeast south southwest west northwest custom percentage</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">reference</td>
 *     <td valign="top">Reference point or number for position calculation.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">null</td>
 *     <td valign="top">integers &quot;x,y&quot;</td>
 *   </tr>
 *
 * </table>
 * <h3>Example</h3>
 * <blockquote><pre>
 * &lt;click
 *    type=&quot;JMenuMouseEventData&quot;
 *    refid=&quot;Menu&quot;
 *    node=&quot;Music&quot;
 * /&gt;
 * </pre></blockquote>
 * <p>
 * The above clicks once on the center of the component identified by
 * Menu.
 * </p>
 * @see junit.extensions.jfcunit.eventdata.JMenuMouseEventData
 * @author Kevin Wilson
 */
public class JMenuMouseEventDataTagHandler extends BaseEventDataTagHandler {
    /**
     * Constructor for JMenuMouseEventDataTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public JMenuMouseEventDataTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#processElement()
     * @throws XMLException provided to complete contract only.
     */
    public void processElement() throws XMLException {
        validateElement();

        JComponent mb = (JComponent) getComponent();

        int[]      indexes = getIndexes();

        if (indexes == null) {
            String path = getPath();

            if (path != null) {
                PathTagHandler pdth = new PathTagHandler(
                        getElement(),
                        getXMLTestCase());
                PathData       pd = pdth.getPathData();
                indexes = pd.getIndexes(mb);
            } else {
                String   pathName = getPathRefid();
                PathData pd = (PathData) getXMLTestCase().getProperty(pathName);
                indexes = pd.getIndexes(mb);
            }
        }

        setEventData(
            new JMenuMouseEventData(
                getJFCTestCase(),
                mb,
                indexes,
                getClicks(),
                getModifiers(),
                getPopupTrigger(),
                getSleepTime()));
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     * @throws XMLException provided to complete contract only.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();
    }

    /**
     * Get the indexes attribute.
     * @return int[] of indexes.
     */
    private int[] getIndexes() {
        String data = getString(INDEXES);

        if (data == null) {
            return null;
        }

        String delimiter = getString(DELIMITER);

        if (delimiter == null) {
            delimiter = ",";
        }

        StringTokenizer tok     = new StringTokenizer(data, delimiter);
        int[]           indexes = new int[tok.countTokens()];
        int             i       = 0;

        while (tok.hasMoreTokens()) {
            String token = tok.nextToken();
            indexes[i++] = Integer.parseInt(token);
        }

        return indexes;
    }

    /**
     * Get the path attribute.
     * @return Path attribute.
     */
    private String getPath() {
        return getString(PATH);
    }

    /**
     * Get the path refid.
     * @return Path Reference ID.
     */
    private String getPathRefid() {
        return getString(PATHREFID);
    }
}
