package junit.extensions.jfcunit.eventdata;

import junit.extensions.jfcunit.JFCTestCase;
import junit.extensions.jfcunit.xml.JFCXMLConstants;

import org.w3c.dom.Element;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import javax.swing.JTabbedPane;
import javax.swing.plaf.basic.BasicTabbedPaneUI;


/**
 * Data container class that holds all the data necessary for jfcUnit to fire mouse events.
 * This class is specific to events on a JTabbedPane.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class JTabbedPaneMouseEventData extends AbstractMouseEventData {
    /**
     * Default value specifying whether the mouse event being fired
     * would trigger a popup or not.
     */
    public static final boolean DEFAULT_ISPOPUPTRIGGER = false;

    /**
     * Default value for the tab title.
     */
    public static final String DEFAULT_TITLE = "";

    /**
     * The JTabbedPane on which to trigger the event.
     */
    private JTabbedPane m_tabPane;

    /**
     * The title of the specific tab on which to trigger the event.
     */
    private String m_title;

    /**
         * The zero-based tab index of the specific tab on which to trigger the event.
     */
    private int m_tabIndex = -1;

    /**
     * Constructor.
     */
    public JTabbedPaneMouseEventData() {
        super();
        setValid(false);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane  The component on which to trigger the event.
     * @param tabIndex The zero-based tab index of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final int numberOfClicks) {
        this(testCase, tabPane, tabIndex, DEFAULT_TITLE, numberOfClicks);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane  The component on which to trigger the event.
     * @param tabIndex The zero-based tab index of the specific tab on which to trigger the event.
     * @param title    The title of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final String title,
        final int numberOfClicks) {
        this(testCase, tabPane, tabIndex, title, numberOfClicks,
            DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane   The component on which to trigger the event.
     * @param tabIndex  The zero-based tab index of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex,
        final int numberOfClicks, final long sleepTime) {
        this(testCase, tabPane, tabIndex, DEFAULT_TITLE, numberOfClicks,
            sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane   The component on which to trigger the event.
     * @param tabIndex  The zero-based tab index of the specific tab on which to trigger the event.
     * @param title     The title of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final String title,
        final int numberOfClicks, final long sleepTime) {
        this(testCase, tabPane, tabIndex, title, numberOfClicks,
            DEFAULT_ISPOPUPTRIGGER, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane  The component on which to trigger the event.
     * @param tabIndex The zero-based tab index of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex,
        final int numberOfClicks, final boolean isPopupTrigger) {
        this(testCase, tabPane, tabIndex, DEFAULT_TITLE, numberOfClicks,
            isPopupTrigger);
    }

    /**
     * Constructor.
     *
     * @param testCase The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane  The component on which to trigger the event.
     * @param tabIndex The zero-based tab index of the specific tab on which to trigger the event.
         * @param title    The title of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                  Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                  boolean specifying whether this event will show a popup.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final String title,
        final int numberOfClicks, final boolean isPopupTrigger) {
        this(testCase, tabPane, tabIndex, title, numberOfClicks,
            isPopupTrigger, DEFAULT_SLEEPTIME);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane   The component on which to trigger the event.
     * @param tabIndex  The zero-based tab index of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex,
        final int numberOfClicks, final boolean isPopupTrigger,
        final long sleepTime) {
        this(testCase, tabPane, tabIndex, DEFAULT_TITLE, numberOfClicks,
            isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane   The component on which to trigger the event.
     * @param tabIndex  The zero-based tab index of the specific tab on which to trigger the event.
         * @param title     The title of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final String title,
        final int numberOfClicks, final boolean isPopupTrigger,
        final long sleepTime) {
        this(testCase, tabPane, tabIndex, title, numberOfClicks,
            getDefaultModifiers(isPopupTrigger), isPopupTrigger, sleepTime);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane   The component on which to trigger the event.
     * @param tabIndex  The zero-based tab index of the specific tab on which to trigger the event.
     * @param title     The title of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final String title,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime) {
        this(testCase, tabPane, tabIndex, title, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, DEFAULT_POSITION, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane   The component on which to trigger the event.
     * @param tabIndex  The zero-based tab index of the specific tab on which to trigger the event.
         * @param title     The title of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final String title,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position) {
        this(testCase, tabPane, tabIndex, title, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, position, null);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane   The component on which to trigger the event.
     * @param tabIndex  The zero-based tab index of the specific tab on which to trigger the event.
         * @param title     The title of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param referencePoint
     *                   The CUSTOM mouse position within the cell.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final String title,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime,
        final Point referencePoint) {
        this(testCase, tabPane, tabIndex, title, numberOfClicks, modifiers,
            isPopupTrigger, sleepTime, CUSTOM, referencePoint);
    }

    /**
     * Constructor.
     *
     * @param testCase  The JFCTestCase on whose thread <code>awtSleep()</code> has to be invoked.
     * @param tabPane   The component on which to trigger the event.
     * @param tabIndex  The zero-based tab index of the specific tab on which to trigger the event.
     * @param title     The title of the specific tab on which to trigger the event.
     * @param numberOfClicks
     *                   Number of clicks in the MouseEvent (1 for single-click, 2 for double clicks)
     * @param modifiers The modifier key values that need to be passed onto the event.
     * @param isPopupTrigger
     *                   boolean specifying whether this event will show a popup.
     * @param sleepTime
     *                   The wait time in ms between each event.
     * @param position  The relative mouse position within the cell.
     * @param referencePoint
     *                                   If position is CUSTOM then the point is a offset from
     *                                   the location of the component. If the position is PERCENT
     *                                   then the location is a percentage offset of the hight and width.
     *                                   Otherwise, the referencePoint is unused.
     * Note: Due to defect in the JRE fopr 1.2.2 and 1.3, the last
     * tab click placement will always be 10 pixels over in the center
     * of the height. The getCellRect in these levels of the JRE return a
     * bogus size for the last tab.
     */
    public JTabbedPaneMouseEventData(final JFCTestCase testCase,
        final JTabbedPane tabPane, final int tabIndex, final String title,
        final int numberOfClicks, final int modifiers,
        final boolean isPopupTrigger, final long sleepTime, final int position,
        final Point referencePoint) {
        setTestCase(testCase);
        setSource(tabPane);
        setNumberOfClicks(numberOfClicks);
        setModifiers(modifiers);
        setPopupTrigger(isPopupTrigger);
        setTabIndex(tabIndex);
        setTitle(title);
        setSleepTime(sleepTime);
        setPosition(position);
        setReferencePoint(referencePoint);
        setValid(true);
    }

    /**
     * Set the attribute value.
     *
     * @param tabPane The new value of the attribute.
     */
    public final void setSource(final JTabbedPane tabPane) {
        m_tabPane = tabPane;
    }

    /**
     * Get the attribute value.
     *
     * @return JTabbedPane    The value of the attribute.
     */
    public final JTabbedPane getSource() {
        return m_tabPane;
    }

    /**
     * The component on which the event has to be fired.
     *
     * @return The component.
     */
    public Component getComponent() {
        // by default, the component is the same as the source
        return getSource();
    }

    /**
     * Set the attribute value.
     *
     * @param tabIndex The new value of the attribute.
     */
    public final void setTabIndex(final int tabIndex) {
        m_tabIndex = tabIndex;
    }

    /**
     * Get the attribute value.
     *
     * @return int    The value of the attribute.
     */
    public final int getTabIndex() {
        return m_tabIndex;
    }

    /**
     * Set the attribute value.
     *
     * @param title The new value of the attribute.
     */
    public final void setTitle(final String title) {
        m_title = title;
    }

    /**
     * Get the attribute value.
     *
     * @return String    The value of the attribute.
     */
    public final String getTitle() {
        return m_title;
    }

    /**
     * Check if this event can consume the event given.
     *
     * @param ae AWTEvent to be consumed.
     * @return true if the event may be consumed.
     */
    public boolean canConsume(final AWTEvent ae) {
        if ((ae.getSource() instanceof JTabbedPane)
                && super.canConsume(ae)
                && sameSource(ae)) {
            MouseEvent me = (MouseEvent) ae;

            if ((me.getID() == MouseEvent.MOUSE_ENTERED)
                    || (me.getID() == MouseEvent.MOUSE_EXITED)) {
                return true;
            }

            JTabbedPane tp    = (JTabbedPane) me.getSource();
            int         index = ((BasicTabbedPaneUI) tp.getUI())
                .tabForCoordinate(
                    tp,
                    me.getX(),
                    me.getY());

            if ((getTabIndex() == -1) || (getTabIndex() == index)) {
                return true;
            }

            return false;
        }

        return false;
    }

    /**
     * Consume the event.
     * @param ae AWTEvent to be consumed.
     * @return boolean true if the event was consumed.
     */
    public boolean consume(final AWTEvent ae) {
        if (super.consume(ae)) {
            return true;
        }

        MouseEvent me = (MouseEvent) ae;

        if (me.getClickCount() == 0) {
            return true;
        }

        JTabbedPane source = (JTabbedPane) me.getSource();
        setSource(source);
        setModifiers(me.getModifiers());
        setNumberOfClicks(me.getClickCount());
        setPopupTrigger(me.isPopupTrigger());

        Point p = new Point(
                me.getX(),
                me.getY());
        Point screen = source.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        int index = ((BasicTabbedPaneUI) source.getUI()).tabForCoordinate(source,
                p.x, p.y);

        if (index != -1) {
            setTitle(source.getTitleAt(index));
        }

        setTabIndex(index);

        setPosition(CENTER);
        setReferencePoint(null);
        setSleepTime(getDefaultSleepTime());

        setValid(true);

        return true;
    }

    /**
     * Compare to event datas and deterime if they are equal.
     *
     * @param o Object to be compared.
     * @return true if the events are the same.
     */
    public boolean equals(final Object o) {
        if (!super.equals(o)) {
            return false;
        }

        JTabbedPaneMouseEventData data = (JTabbedPaneMouseEventData) o;

        return (data.getTabIndex() == getTabIndex())
            && (data.getTitle() == getTitle());
    }

    /**
     * Get the hashCode of this object.
     * @return int hashCode.
     */
    public int hashCode() {
        return super.hashCode();
    }

    /**
     * Populate the XML Element with this object attributes.
     * @param e element to be populated.
     */
    public void populate(final Element e) {
        super.populate(e);
        e.setAttribute(JFCXMLConstants.TYPE, "JTabbedPaneMouseEventData");
        e.setAttribute(JFCXMLConstants.INDEX, "" + getTabIndex());

        /* @todo Only one of Index or Title is allowed.
        if (getTitle() != null) {
          e.setAttribute(JFCXMLConstants.TITLE, getTitle());
        }*/
    }

    /**
     * Prepare the component to receive the event.
     *
     * @return true if the component is ready to receive the event.
     */
    public boolean prepareComponent() {
        if (!isValidForProcessing(getSource())) {
            return false;
        }

        // if the tab is not currently visible
        if (m_tabIndex < 0) {
            m_tabIndex = m_tabPane.indexOfTab(getTitle());
        }

        if (m_tabPane.getBoundsAt(m_tabIndex) == null) {
            return false;
        }

        JFCTestCase testCase = getTestCase();

        if (testCase != null) {
            // try to clear the event queue
            testCase.pauseAWT();
        }

        Rectangle tabPaneRect = m_tabPane.getBoundsAt(m_tabIndex);

        // Adjust for java 1.2 and 1.3 bug where the last tab
        // panes rect is not returning the correct size.
        if ((getPosition() != EventDataConstants.CUSTOM)
                && (m_tabIndex == (m_tabPane.getTabCount() - 1))) {
            // Use a positional calculation.
            setPosition(EventDataConstants.CUSTOM);

            // 10 pixels from the left and
            // middel of the height
            setReferencePoint(new Point(10, tabPaneRect.height / 2));
        }

        if (testCase != null) {
            // try to clear the event queue
            testCase.flushAWT();
        }

        Point p = calculatePoint(tabPaneRect);

        if (!m_tabPane.getVisibleRect().contains(p)) {
            Rectangle vis     = m_tabPane.getVisibleRect();
            Rectangle newView = new Rectangle(p.x - (int) (vis.width / 2),
                    p.y - (int) (vis.height / 2), vis.width, vis.height);
            m_tabPane.scrollRectToVisible(newView);
        }

        Point screen = m_tabPane.getLocationOnScreen();
        screen.translate(p.x, p.y);
        setLocationOnScreen(screen);

        return true;
    }

    /**
     * Get string description of event.
     *
     * @return String description of event.
     */
    public String toString() {
        if (!isValid()) {
            return super.toString();
        }

        StringBuffer buf = new StringBuffer(1000);
        buf.append(super.toString());
        buf.append(" title: " + getTitle());
        buf.append(" index: " + getTabIndex());

        return buf.toString();
    }
}
