package junit.extensions.jfcunit.finder;

import java.awt.Component;


/**
 * This is a 'breakaway' from the ComponentFinder in which the component to be
 * tested is assumed to have a name and so the name passed in is compared to the comp.name
 * The pattern syntax can be found at the Jakarta RegExp API Documentation in {@link org.apache.regexp.RE}.
 *
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class NamedComponentFinder extends Finder {
    /**
     * The type of the component.
     */
    private Class m_compCls = null;

    /**
     * The pattern for the name of the component.
     */
    private String m_name = null;

    /**
     * A boolean specifying whether the filtration
     * is case insensitive.
     */
    private boolean m_caseIndependent = false;

    /**
     * Constructor accepting all arguments needed to
     * filter component.
     *
     * @param cls    The desired type of the component.
     * @param name    The desired pattern for the name of the component.
     */
    public NamedComponentFinder(final Class cls, final String name) {
        this(cls, name, false);
    }

    /**
     * Constructor accepting all arguments needed to filter
     * component.
     *
     * @param cls                The desired type of the component.
     * @param name               The desired pattern for the name of the component.
     * @param caseIndependent    Whether the match should be case independent (true) or not (false)
     */
    public NamedComponentFinder(final Class cls, final String name,
        final boolean caseIndependent) {
        setComponentClass(cls);
        setName(name);
        this.m_caseIndependent     = caseIndependent;
        recreatePatternMatcher(m_name, caseIndependent);
    }

    /**
     * Set the component class.
     * @param cls Class to be found by the finder.
     */
    public final void setComponentClass(final Class cls) {
        m_compCls = cls;
    }

    /**
     * Get the component class to be found by the finder.
     * @return Class to be found.
     */
    public final Class getComponentClass() {
        return m_compCls;
    }

    /**
     * Set the name of the component to be found.
     * @param name String name of the component.
     */
    public final void setName(final String name) {
        m_name = name;
        recreatePatternMatcher(m_name, m_caseIndependent);
    }

    /**
     * Set the finder into a case independent mode.
     * @param ignoreCase true if case should be ignored.
     */
    public void setCaseIndependent(final boolean ignoreCase) {
        super.setCaseIndependent(ignoreCase);
        m_caseIndependent = ignoreCase;
        recreatePatternMatcher(m_name, m_caseIndependent);
    }


    /**
     * Get the name of the component to be found.
     * @return String name of the component.
     */
    public final String getName() {
        return m_name;
    }

    /**
     * Method that returns true if the given component matches the search
     * criteria.
     *
     * @param comp   The component to test.
     * @return true if this component is a match.
     */
    public boolean testComponent(final Component comp) {
        return ((comp != null)
        && ((m_compCls == null) || isValidForProcessing(comp, m_compCls))
        && evaluate(
            comp.getName(),
            m_name));
    }
}
