package junit.extensions.jfcunit.finder;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import org.w3c.dom.Element;


/**
 * This is a extension of the findTagHandler element. Where the
 * finder specified is NamedComponentFinder.
 *
 * <h3>Description</h3>
 * <p>
 *   This creates a NamedComponentFinder and executes the find.
 * </p>
 * <h3>Finder tag name</h3>
 * <p>
 * NamedComponentFinder
 * </p>
 * <h3>Parameters</h3>
 * <table border="1" cellpadding="2" cellspacing="0">
 *   <tr>
 *     <td valign="top"><b>Attribute</b></td>
 *     <td valign="top"><b>Description</b></td>
 *     <td align="center" valign="top"><b>Required</b></td>
 *     <td valign="top"><b>Default</b></td>
 *     <td valign="top"><b>Values</b></td>
 *   </tr>
 *   <tr>
 *     <td valign="top">finder</td>
 *     <td valign="top">For this element the type must be set to NamedComponentFinder</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">N/A</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">id</td>
 *     <td valign="top">Id for the object found.</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">String</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">refid</td>
 *     <td valign="top">Id for the object found.</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">Search in all windows</td>
 *     <td valign="top">Panel to be searched</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">index</td>
 *     <td valign="top">The index of the element to select from the combo box.</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">N/A</td>
 *     <td valign="top">Zero based index of list items.</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">name</td>
 *     <td valign="top">The name of the component to be found.</td>
 *     <td valign="top" align="center">Yes</td>
 *     <td valign="top">None</td>
 *     <td valign="top">Name of the component.</td>
 *   </tr>
 *   <tr>
 *     <td valign="top">caseindependent</td>
 *     <td valign="top">Ignore the case if true</td>
 *     <td valign="top" align="center">No</td>
 *     <td valign="top">false</td>
 *     <td valign="top">true/false</td>
 *   </tr>
 * </table>
 * <h3>Example</h3>
 * <blockquote><pre>
 * &lt;find
 *    finder=&quot;NamedComponentFinder&quot;
 *    id=&quot;NamedComponent&quot;
 *    class=&quot;javax.swing.JButton&quot;
 *    name=&quot;MyName&quot;
 *    index=&quot;0&quot;
 * /&gt;
 * </pre></blockquote>
 * <p>
 * The above finds the JButton identified &quot;MyName&quot;
 * </p>
 * @see junit.extensions.jfcunit.finder.NamedComponentFinder
 * @author Kevin Wilson
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class NamedComponentFinderTagHandler extends BaseFindTagHandler {
    /**
     * Constructor for FindTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public NamedComponentFinderTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#processElement()
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public void processElement() throws XMLException {
        validateElement();

        String  name            = getName();
        String  clazzName       = getClassName();
        boolean caseIndependent = getCaseIndependent();

        try {
            Class clazz = Class.forName(clazzName);
            find(new NamedComponentFinder(clazz, name, caseIndependent));
        } catch (ClassNotFoundException ex) {
            throw new XMLException("Could not find class:" + clazzName, ex,
                getElement(),
                getXMLTestCase().getPropertyCache());
        }
    }

    /**
     * @see junit.extensions.xml.elements.AbstractTagHandler#validateElement()
     * @throws XMLException when errors encountered processing XML.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // reqd attribute: name
        checkRequiredAttribute(NAME);

        // reqd attribute: class
        checkRequiredAttribute(CLASS);
    }
}
