package junit.extensions.xml;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;


/**
 * <p>Title: XMLObjectCache</p>
 * <p>Description: Multi level hashmap to contain
 * name value pairs. A parent level can be explicitly
 * accessed by prefixing the name with "../"
 * otherwise the parent level will only be searched
 * if the object does not exist at the current level. </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: jfcunit project</p>
 * @author Kevin Wilson
 * @version 1.0
 */
public class XMLObjectCache {
    /**
     * Internal hashmap used to hold the property/procedure mappings.
     */
    private final HashMap m_map = new HashMap();

    /**
     * The parent XMLObjectCache.
     */
    private XMLObjectCache m_parent;

    /**
     * Empty contructor. Parent is assumed to be null. It can be
     * set at a later time.
     */
    public XMLObjectCache() {
        this(null);
    }

    /**
     * Constructor.
     * @param parent Parent object cache to be set.
     */
    public XMLObjectCache(final XMLObjectCache parent) {
        setParent(parent);
    }

    /**
     * Do a reverse lookup of a mapping.
     * Return the name of a object based upon the value.
     * @param value Value to return the name for.
     * @return The first matching name is returned.
     */
    public final String getName(final Object value) {
        String ret = null;

        if (m_map.containsValue(value)) {
            Iterator entries = m_map.entrySet().iterator();

            while (entries.hasNext() && (ret == null)) {
                Map.Entry entry = (Map.Entry) entries.next();

                if (entry.getValue().equals(value)) {
                    ret = (String) entry.getKey();
                }
            }
        }

        if ((m_parent != null) && (ret == null)) {
            ret = m_parent.getName(value);
        }

        return ret;
    }

    /**
     * Get all of the names of the objects currently
     * in the cache. Parent objects will be prefixed
     * with the ../
     *
     * @return list of names.
     */
    public final String[] getNames() {
        int      size        = m_map.size();
        int      parentSize  = 0;
        String[] parentNames;

        if (m_parent == null) {
            parentNames = new String[0];
        } else {
            parentNames = m_parent.getNames();
        }

        String[] names = new String[parentNames.length + size];
        int      idx  = 0;
        Iterator keys = m_map.keySet().iterator();

        while (keys.hasNext()) {
            names[idx++] = (String) keys.next();
        }

        for (int i = 0; i < parentNames.length; i++) {
            names[idx++] = "../" + parentNames[i];
        }

        return names;
    }

    /**
     * Set the parent object cache.
     * @param parent Parent object cache to be traversed by
     * get and explicitly traversed by put/remove.
     */
    public final void setParent(final XMLObjectCache parent) {
        // Move any pre-existing keys that have been
        // defined at the parent level before the parent
        // was set to the parents cache.
        if ((parent != null) && (m_map.size() > 0)) {
            // Move keys with ../ prefix to parent.
            Vector   moves = new Vector();
            Iterator iter = m_map.keySet().iterator();

            while (iter.hasNext()) {
                String key = (String) iter.next();

                if (key.startsWith("../")) {
                    moves.add(key);
                }
            }

            iter = moves.iterator();

            while (iter.hasNext()) {
                String key   = (String) iter.next();
                Object value = m_map.remove(key);
                key = key.substring(3);
                parent.put(key, value);
            }
        }

        m_parent = parent;
    }

    /**
     * Get the parent object cache.
     * @return Parent object cache.
     */
    public final XMLObjectCache getParent() {
        return m_parent;
    }

    /**
     * Clear the cache.
     */
    public final void clear() {
        m_map.clear();
    }

    /**
     * Put a new mapping into the cache. If the
     * name starts with ../ then place the object
     * in the parent cache.
     *
     * @param name Name of the property or procedure.
     * @param value Value of the property or procedure.
     */
    public final void put(final String name, final Object value) {
        // Navigate up a level
        if ((m_parent != null) && name.startsWith("../")) {
            m_parent.put(
                name.substring(3),
                value);

            return;
        }

        if (name.startsWith("./")) {
            m_map.put(
                name.substring(2),
                value);
        } else {
            m_map.put(name, value);
        }
    }

    /**
     * Remove the object from the cache.
     * If the name starts with ../ then propagate the
     * remove to the parent.
     *
     * @param name Name of the object in the cache.
     */
    public final void remove(final String name) {
        if ((m_parent != null) && name.startsWith("../")) {
            m_parent.remove(name.substring(3));
        }

        if (name.startsWith("./")) {
            m_map.remove(name.substring(2));
        } else {
            m_map.remove(name);
        }
    }

    /**
     * Get a object from the cache. If the object does not
     * exist in the cache then try the parent cache. If a parent
     * cache does not exist then try the System properties.
     * @param name Name of the value to be retrieved.
     * @return Return the value.
     */
    public Object get(final String name) {
        if ((name != null) && (m_parent != null) && name.startsWith("../")) {
            return m_parent.get(name.substring(3));
        }

        boolean searchParent = true;
        Object  object;

        if ((name != null) && name.startsWith("./")) {
            object           = m_map.get(name.substring(2));
            searchParent     = false;
        } else {
            object = m_map.get(name);
        }

        if ((object != null) || !searchParent) {
            return object;
        }

        if (m_parent != null) {
            return m_parent.get(name);
        }

        return null;
    }
}
