package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestCase;
import junit.extensions.xml.XMLException;

import junit.framework.Assert;

import org.w3c.dom.Element;


/**
 * This class will handle the processing of &lt;assertenabled&gt; nodes.
 *
 * <H3>Mandatory attributes</H3>
 *    refid is assumed to be a java.awt.Component.
 *
 * <H3>Optional attributes (default value)</H3>
 *    message Message to be displayed upon assertion.<br>
 *    focus (true)
 *
 * @author <a href="mailto:kwilson227@users.sourceforge.net">Kevin Wilson</a>
 */
public class AssertHasFocusTagHandler extends AbstractAssertTagHandler {
    /**
     * Constructor for AssertEnabledTagHandler.
     *
     * @param element     The element to be processed
     * @param testCase    The IXMLTestCase that uses this element
     */
    public AssertHasFocusTagHandler(final Element element,
        final IXMLTestCase testCase) {
        super(element, testCase);
    }

    /**
     * Obtains the component via the refid. The
     * checks the focus state of the component.
     * If the focus state does not match the state
     * specified then assert.
     *
     * @throws XMLException upon assertion.
     */
    public void processElement() throws XMLException {
        validateElement();

        boolean result = false;
        Object  object = getXMLTestCase().getProperty(getRefId());
        Assert.assertNotNull(
            getMessage(),
            object);

        if (object instanceof java.awt.Component) {
            result = ((java.awt.Component) object).hasFocus();
        } else {
            throw new XMLException("Not a component", null,
                getElement(),
                getTest().getProcedureCache());
        }

        Assert.assertTrue(
            getMessage(),
            result == getFocus());
    }

    /**
     * Insure that the refid is specified.
     * @throws XMLException if refid is not specified.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // reqd attribute: refid
        checkRequiredAttribute(REFID);
    }

    /**
     * Returns the value of the "focus" attribute for this element.
     * @return String  The value of the FOCUS attribute.
     */
    private boolean getFocus() {
        return getBoolean(FOCUS, true);
    }
}
