package junit.extensions.xml.elements;

import junit.extensions.xml.IXMLTestSuite;
import junit.extensions.xml.XMLException;
import junit.extensions.xml.XMLTestSuite;

import org.w3c.dom.Element;


/**
 * This class will handle the processing of &lt;suite&gt; nodes.
 * These nodes may contain taghandlers, procedures, property, test and file
 * elements.<p>
 *
 * Procedures defined at the test suite level are visible to all
 * children test cases. Children may define procedures of the same
 * name. Access to the higher level procedure is then done via
 * explicit reference to the parent with "../".<p>
 *
 * <H3>Required Attributes</H3>
 * name - The test suite name.
 *
 * <H3>Children</H3>
 * <pre>
 * file - see FileTagHander
 * taghandlers - see TagHandlersTagHander
 * suite - recursive reference.
 * test - see TestTagHandler
 * procedure - see ProcedureTagHandler
 * property - see PropertyTagHandler
 * </pre>
 * @author <a href="mailto:vraravam@thoughtworks.com">Vijay Aravamudhan : ThoughtWorks Inc.</a>
 */
public class SuiteTagHandler extends AbstractTagHandler {
    /**
     * Default constructor.
     *
     * @param element      The element to be processed
     * @param testSuite    The XMLTestSuite that uses this element
     */
    public SuiteTagHandler(final Element element, final IXMLTestSuite testSuite) {
        super(element, testSuite);
        validateElement();
        addTestSuite();
    }

    /**
     * Process the children of the suite.
     * @throws XMLException is thrown if the element cannot be understood.
     */
    public void processElement() throws XMLException {
    }

    /**
     * Validate that the tag name is suite.
     * @throws XMLException if this is used against a
     * different tag.
     */
    public void validateElement() throws XMLException {
        // do the default validations from the super class
        super.validateElement();

        // check the element tag name
        checkElementTagName(SUITE);
        checkRequiredAttribute(NAME);
    }

    /**
     * Get the value of the name attribute.
     * @return String value of the name attribute.
     */
    protected final String getName() {
        return getString(NAME);
    }

    /**
     * Add the test suite to the parent XMLTestSuite.
     */
    protected void addTestSuite() {
        IXMLTestSuite parent = getXMLTestSuite();
        parent.addTest(createSuite(
                getName(),
                getElement()));
    }

    /**
     * Creates a XMLTestSuite implementation.
     * @param file file to be processed.
     * @param element element to be processed.
     * @return IXMLTestSuite Impementation.
     */
    protected IXMLTestSuite createSuite(final String file, final Element element) {
        return new XMLTestSuite(file, element);
    }
}
